#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "file_manip.h"
#include <errno.h>
#include <sys/stat.h>

#define BUFFER_SIZE 20

/* Gestion des erreurs dans les arguments */
void arguments_error(char* message)
{
  fprintf(stderr,"Erreur %s.\n", message);
  fprintf(stderr,"Usage: mon_cp source destination\n");
  exit(1);
}


/* Fonction utile à partir de l'exercice 4 pour déterminer le nom du
   fichier destination */
char * create_copy_name(char* source, char* destination)
{
  char * basename = strrchr(source,'/');
  if ( NULL == basename ) {
    basename = source;
  }
  char * dirname = destination;
  char * filename =
    (char*) malloc(sizeof(char) * (strlen(basename) + 1 + strlen(dirname) + 1));
  if ( NULL == filename ) {
    return NULL;
  }
  *filename='\0';
  strcat(filename,dirname);
  strcat(filename,"/");
  strcat(filename,basename);
  return filename;
}

/* Copie de fichier */
int copy(char* source, char* destination)
{
  printf("Copie du fichier %s sur le fichier %s\n",source,destination);

  // Ouverture des fichiers source et destination respectivement en
  // lecture et en écriture
  MY_FILE* src = my_fopen(source, "r");
  if ( NULL == src ) {
    perror("my_fopen");
    return 1;
  }
  MY_FILE* dst = my_fopen(destination, "w");
  if ( NULL == dst ) {
    perror("my_fopen");
    my_fclose(src);
    return 1;
  }

  // Allocation de la mémoire tampon pour transférer des données du
  // fichier source sur le fichier destination
  char* buf = (char*) malloc(sizeof(char) * BUFFER_SIZE);
  if ( NULL == buf ) {
    perror("malloc");
    my_fclose(src);
    my_fclose(dst);
    return 1;
  }

  int res = 0;
  int nb_data_read = 0;

  // On boucle tant qu'il y a quelque chose à lire.
  // On copie ensuite ce qui vient d'être lu dans le fichier destination.
  // On prend soin de ne copier que ce qui vient d'être lu :
  // (nb_data_read != BUFFER_SIZE) !
  errno = 0;
  while ( (nb_data_read = my_fread(buf, sizeof(char), BUFFER_SIZE, src)) ) {
    res = my_fwrite(buf, sizeof(char), nb_data_read, dst);

    // On verifie les erreurs eventuelles pour my_fwrite
    if ( res != nb_data_read ) {
      perror("my_fwrite");
      my_fclose(src);
      my_fclose(dst);
      free(buf);
      return 1;
    }
  }

  // On verifie les erreurs eventuelles sur errno pour my_fread
  if (errno) {
    perror("my_fread failed");
    my_fclose(src);
    my_fclose(dst);
    free(buf);
    return 1;
  }

  // On ferme les fichiers source et destination
  my_fclose(src);
  if (errno) {
    perror("my_fclose");
  }
  my_fclose(dst);
  if (errno) {
    perror("my_fclose");
  }

  // On libere la mémoire occupée par le buffer
  free(buf);
  return 0;
}


/* Programme principal */
int main(int argc, char* argv[])
{
  int error_code = 0;
  /* Gestion des paramètres */
  char * source;
  char * destination;
  struct stat stat_source;
  struct stat stat_destination;
  switch(argc) {
    case 3:
      /* cas des exercices 1 à 3 */
      /* des modifications sont requises pour l'exercice 4 */
      source = argv[1];
      destination = argv[2];

      if ( -1 == stat(source, &stat_source) ) {
	perror("stat on source");
	return 1;
      }
      if ( !S_ISREG(stat_source.st_mode) ) {
	arguments_error("Le fichier source doît être un fichier régulier");
      }

      if ( -1 == stat(destination, &stat_destination) ) {
	if ( ENOENT != errno ) {
	  perror("stat on destination");
	  return 1;
	}
      }
      else if ( (stat_destination.st_dev == stat_source.st_dev) &&
	        (stat_destination.st_ino == stat_source.st_ino) ) {
	arguments_error("Les fichiers source et destination sont identiques");
      }
      else if ( S_ISDIR(stat_destination.st_mode) ) {
	if (NULL == (destination = create_copy_name(source, destination)) ) {
	  perror("create_copy_name");
	  return 1;
	}
      }
      error_code = copy(source, destination);
      break;
    default:
      arguments_error("nombre d'arguments invalide");
    }
  return error_code;
}
