#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

/* #define DEBUG */
/* #define DEBUG_ENTER */
/* #define DEBUG_EXIT */
#define DEBUG_STAT

typedef unsigned char byte;

struct block {
  size_t block_size;
  size_t used_size;
  struct block *next;
  int used;
};

struct block *first = NULL;

struct block *new_block(size_t size) {
  struct block *base;
#ifdef DEBUG
  fprintf(stderr,"**** NEW BLOCK ****\n");
#endif
  base = (struct block *)sbrk((int) (size + sizeof(struct block)));
  if (base == (struct block *) -1) {
    perror ("sbrk");
    return NULL;
  }
  base->next = NULL;
  base->used = 0;
  base->block_size = size;
  return base;
}


void *mmalloc (size_t size) {
  struct block *pblock;
  struct block *prev_pblock;

  if ( NULL == first ) {
    if ( NULL == (first = new_block(size)) ) {
      return NULL;
    }
    first->used = 1;
    first->used_size = size;
    return (void *) (first+1);
  }

  pblock = first;
  prev_pblock = first;
  while (NULL != pblock) {
    if ( !pblock->used && pblock->block_size >= size ) {
#ifdef DEBUG
      fprintf(stderr,"**** OLD BLOCK ****\n");
#endif
      pblock->used = 1;
      pblock->used_size = size;
      return (void *) (pblock+1);
    }
    prev_pblock = pblock;
    pblock = pblock->next;
  }

  /* Il n'y a pas de block libre */
  if ( NULL == (pblock = new_block(size)) ) {
    return NULL;
  }
  prev_pblock->next = pblock;
  pblock->used = 1;
  pblock->used_size = size;
  return (void *) (pblock+1);
}

void mfree(void *p) {
  struct block *pb;
  if ( NULL != p ) {
    pb = ((struct block*) p) - 1;
    pb->used = 0;
  }
}

void * mrealloc(void *p, size_t size) {
  struct block *pb;

  if ( NULL == p ) {
    /* allocation d'une nouvelle zone */
    return mmalloc(size);
  }

  if ( 0 == size ) {
    /* desallocation */
    mfree(p);
    return NULL;
  }

  pb = ((struct block*) p) - 1;

  if ( size <= pb->block_size ) {
    /* reutilisation de la zone memoire */
    pb->used_size = size;
    return p;
  }

  {
  /* utilisation d'une nouvelle zone memoire */
    void *new_p;
    new_p = mmalloc(size);
    if ( NULL != new_p ) {
      /* copie memoire */
      size_t i;
      byte *bp = (byte *) p;
      byte *new_bp = (byte *) new_p;
      for (i = 0; i < pb->used_size; i++) {
	new_bp[i] = bp[i];
      }
    }
    mfree(p);
    return new_p;
  }
}

void * mcalloc(size_t count, size_t size) {
  void *p = mmalloc(count * size);
  byte *bp = (byte *) p;
  if ( NULL != p ) {
    size_t i;
    size_t max = count * size;
    for (i = 0; i < max; i++) {
      bp[i] = 0;
    }
  }
  return p;
}

/* ---------------- */

void print_stat () {
  int nb_blocks = 0;
  int nb_free_blocks = 0;
  int internal_lost = 0;
  struct block * pblock = first;

  while ( NULL != pblock ) {
    nb_blocks++;
    if ( pblock->used ) {
      internal_lost = internal_lost + (pblock->block_size - pblock->used_size);
    } else {
      nb_free_blocks++;
    }
    pblock = pblock->next;
  }

  fprintf(stderr, "--------------\n");
  fprintf(stderr, "number of blocks:\t%d\n", nb_blocks);
  fprintf(stderr, "number of free blocks:\t%d\n", nb_free_blocks);
  fprintf(stderr, "lost memory inside allocated blocks:\t%d\n", internal_lost);
}

/* ---------------- */

void *malloc (size_t size) {
  void *p;
#ifdef DEBUG_ENTER
  fprintf(stderr,"-> enter malloc\n");
#endif
  p = mmalloc(size);
#ifdef DEBUG_EXIT
  fprintf(stderr,"-> exit malloc\n");
#endif
#ifdef DEBUG_STAT
  print_stat();
#endif
  return p;
}

void free(void *p) {
#ifdef DEBUG_ENTER
  fprintf(stderr,"-> enter free\n");
#endif
  mfree(p);
#ifdef DEBUG_EXIT
  fprintf(stderr,"-> exit free\n");
#endif
}

void * realloc(void *ptr, size_t size) {
  void *p;
#ifdef DEBUG_ENTER
  fprintf(stderr,"-> enter realloc\n");
#endif
  p = mrealloc(ptr, size);
#ifdef DEBUG_EXIT
  fprintf(stderr,"-> exit realloc\n");
#endif
#ifdef DEBUG_STAT
  print_stat();
#endif
  return p;
}

void * calloc(size_t count, size_t size) {
  void *p;
#ifdef DEBUG_ENTER
  fprintf(stderr,"-> enter calloc\n");
#endif
  p = mcalloc(count, size);
#ifdef DEBUG_EXIT
  fprintf(stderr,"-> exit calloc\n");
#endif
#ifdef DEBUG_STAT
  print_stat();
#endif
  return p;
}
