// -*- mode: c++; c-indent-level: 4; c++-member-init-indent: 8; comment-column: 35; -*-

//-----------------------------------------------------------------------------
// eoSSPSO.h
// (c) OPAC 2007
/*
    Contact: paradiseo-help@lists.gforge.inria.fr
 */
//-----------------------------------------------------------------------------

#ifndef _EOSSPSO_H
#define _EOSSPSO_H

//-----------------------------------------------------------------------------
#include <eoPSO.h>
#include <eoContinue.h>
#include <eoStandardFlight.h>
#include <eoStarTopology.h>
#include <eoStandardVelocity.h>
#include <utils/eoRealVectorBounds.h>
#include <eoRealBoundModifier.h>
//-----------------------------------------------------------------------------

/**
 * A Star ("S"tar topology) Standard ("S"tandard velocity) PSO.
 * You can use it with or without bounds on the velocity.
 * No bound for the flight (no bounds for the positions).
 *
 */
template < class POT > class eoSSPSO:public eoPSO < POT >
{
public:


    typedef typename POT::ParticleVelocityType VelocityType;

    /** Full constructor
    * @param _continuator - An eoContinue that manages the stopping criterion and the checkpointing system
    * @param _eval - An eoEvalFunc: the evaluation performer
    * @param _c1 - The first learning factor used for the particle's best. Type must be POT::ParticleVelocityType 
    * @param _c2 - The second learning factor used for the local/global best(s). Type must be POT::ParticleVelocityType 
    * @param _bounds - An eoRealBaseVectorBounds: real bounds for real velocities. 
    * If the velocities are not real, they won't be bounded by default. Should have a eoBounds ?
    * @param _boundsModifier - An eoRealBoundModifier used to modify the bounds (for real bounds only)
    */
    eoSSPSO (
        eoContinue < POT > &_continuator,
        eoEvalFunc < POT > &_eval,
        const VelocityType & _c1,
        const VelocityType & _c2 ,
        eoRealVectorBounds & _bounds,
        eoRealBoundModifier & _bndsModifier):
            continuator (_continuator),
            eval (_eval),
            velocity(eoStandardVelocity<POT>(topology,_c1,_c2,_bounds,_bndsModifier)),
            bounds(_bounds),
            boundsModifier(_bndsModifier)
    {}

    /** Constructor without bound modifier.
    * @param _continuator - An eoContinue that manages the stopping criterion and the checkpointing system
    * @param _eval - An eoEvalFunc: the evaluation performer
    * @param _c1 - The first learning factor used for the particle's best. Type must be POT::ParticleVelocityType 
    * @param _c2 - The second learning factor used for the local/global best(s). Type must be POT::ParticleVelocityType 
    * @param _bounds - An eoRealBaseVectorBounds: real bounds for real velocities. 
    * If the velocities are not real, they won't be bounded by default. Should have a eoBounds ?
    */
    eoSSPSO (
        eoContinue < POT > &_continuator,
        eoEvalFunc < POT > &_eval,
        const VelocityType & _c1,
        const VelocityType & _c2 ,
        eoRealVectorBounds & _bounds):
            continuator (_continuator),
            eval (_eval),
            velocity(eoStandardVelocity<POT>(topology,_c1,_c2,_bounds)),
            bounds(_bounds),
            boundsModifier(dummyModifier)
    {}


    /** Constructor without bounds nor bound modifier.
    * @param _continuator - An eoContinue that manages the stopping criterion and the checkpointing system
    * @param _eval - An eoEvalFunc: the evaluation performer
    * @param _c1 - The first learning factor used for the particle's best. Type must be POT::ParticleVelocityType 
    * @param _c2 - The second learning factor used for the local/global best(s). Type must be POT::ParticleVelocityType  
    * If the velocities are not real, they won't be bounded by default. Should have a eoBounds ?
    */
    eoSSPSO (
        eoContinue < POT > &_continuator,
        eoEvalFunc < POT > &_eval,
        const VelocityType & _c1,
        const VelocityType & _c2):
            continuator (_continuator),
            eval (_eval),
            velocity(eoStandardVelocity<POT>(topology,_c1,_c2)),
            bounds(*(new eoRealVectorNoBounds(0))),
            boundsModifier(dummyModifier)
    {}


    /// Apply a few iteration of flight to the population (=swarm).
    virtual void operator  () (eoPop < POT > &_pop)
    {
        try
        {
            // setup the topology (done once)
            topology.setup(_pop);

            do
            {
                // loop over all the particles for the current iteration
                for (unsigned idx = 0; idx < _pop.size (); idx++)
                {
                    // perform velocity evaluation
                    velocity (_pop[idx],idx);

                    // apply the flight
                    flight (_pop[idx]);

                    // evaluate the position
                    eval (_pop[idx]);

                    // update the topology (particle and the global bests)
                    velocity.updateNeighborhood(_pop[idx],idx);
                }
            } while (continuator (_pop));

        }catch (std::exception & e)
        {
            std::string s = e.what ();
            s.append (" in eoSSPSO");
            throw std::runtime_error (s);
        }
    }

protected:
    eoContinue < POT > &continuator;
    eoEvalFunc < POT > &eval;

    eoStandardVelocity < POT > velocity;
    eoStandardFlight < POT > flight;
    eoStarTopology<POT> topology;

    eoRealVectorBounds bounds; // REAL bounds even if the velocity could be of another type.
    eoRealBoundModifier & boundsModifier;

    // If the bound modifier doesn't need to be used, use the dummy instance
    eoDummyRealBoundModifier dummyModifier;

};


#endif /*_EOSSPSO_H*/
